/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package viewer;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URISyntaxException;

import com.inet.report.EngineRenderData;
import com.inet.viewer.Progress;
import com.inet.viewer.RenderData;
import com.inet.viewer.ReportView;
import com.inet.viewer.SwingReportViewer;

/**
 * This is a sample which shows how to keep track of any progresses running in the viewer. The idea is to register as a
 * property change listener either with the Progress object itself or globally with the SwingReportViewer.
 * <p>
 * This listener will then receive any status changes to the progresses and thereby we can keep track of when a progress
 * is done. This is an improved situation over having to poll the viewer consistently whether it is still busy or not.
 */
public class ProgressTracker implements PropertyChangeListener {

    private RenderData        renderConnection;  // This is our main render data connection - the source of our raw report data coming from our report server

    private SwingReportViewer viewer;            // Our top-level viewer object

    private ReportView        currentReportView; // Our report view, responsible for a single report

    /**
     * Track any progress running in the viewer.
     */
    public ProgressTracker() {
        // to initialize we first create a top level ReportViewer:
        viewer = new SwingReportViewer();

        String reportLocation = "";
        try {
            reportLocation = getClass().getResource( "../sample.rpt" ).toURI().toString();
        } catch( URISyntaxException e ) {
            // Nothing to do
        }

        // then initialize the render data connection.
        renderConnection = new EngineRenderData( "report=" + reportLocation );

        // If you use your own i-net Clear Reports server then use the report URL for this server, e.g.:
        // renderConnection = new URLRenderData( "http://server:port?report=" + loc );
        // renderConnection.setReportURL ( "http://serverName:9000/?report=file:c:/java/sample.rpt" );

        // addNewReportView causes a new report view to be created using the given connection as its data source, and then
        // for this newly created report to be added to the viewer.
        currentReportView = viewer.addNewReportView( renderConnection );

        // We now register ourselves as a propertyChangeListener with the viewer so we can keep track of any changes in progresses
        viewer.addStateChangeListener( this );

        currentReportView.print( 1, 1, true );
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new ProgressTracker();
    }

    /**
     * Gets called when a bound property is changed.
     * @param evt A PropertyChangeEvent
     */
    @Override
    public void propertyChange( PropertyChangeEvent evt ) {
        if( evt.getSource() instanceof Progress ) {
            // If a progress has changed...
            Progress p = (Progress)evt.getSource();
            if( p.getType() == Progress.TYPE_PRINT ) {
                // ...and it is the printing progress (the one we started in the constructor)...
                if( evt.getPropertyName().equals( Progress.PROP_PROGRESS_STATUS ) ) {
                    // ...and its status has somehow changed, let's see what status the progress now has!
                    int status = ((Integer)evt.getNewValue()).intValue();
                    switch( status ) {
                        case Progress.STATUS_CANCELED:
                            System.err.println( "Printing canceled!" );
                            break;
                        case Progress.STATUS_COMPLETED:
                            System.err.println( "Printing complete!" );
                            // we're done
                            System.exit( 0 );
                            break;
                        case Progress.STATUS_ERROR:
                            System.err.println( "Printing aborted because of an error!" );
                            // we're done
                            System.exit( 0 );
                            break;
                        case Progress.STATUS_INITIALIZED:
                            System.err.println( "Printing initialized..." );
                            break;
                        case Progress.STATUS_RUNNING:
                            System.err.println( "Printing..." );
                            break;
                        default:
                            break;
                    }
                }
            }
        }
    }
}
